<?php

/**
 * RoboFile
 * Clase para ejecucion de task automaticas
 *
 * @package robo
 * @author Modulos Empresarios / Egytca
 * @author Matias Micheltorena
 * @copyright Copyright (c) 2017, Egytca
 *
 * @see http://robo.li/
 */

require_once('classes/includes/assoc_array2xml.php');
require_once('classes/includes/Egytca/EmailManagement.php');
require_once('classes/includes/Egytca/PropelErrorParser.php');
require_once('classes/includes/Egytca/ConfigXmlManagement.php');

//use Egytca\PropelErrorParser;
//use Egytca\ConfigXmlManagement;

class RoboFile extends \Robo\Tasks {

	private $rootPath;
	private $backupPath;
	private $propelBuildPath;
	private $deployPath;
	private $logPath;

	private $debugMode = false;

	//carpeta donde van a estar los archivos para actualizar
	private $updatedFilesFolder = "/public_html";
	private $sqlBuildScriptFile = "/sql/buildScript.sql";

	private $appData = [];
	private $actions = [];

	private $arrayConverter;
	private $errorParser;
	private $timestampFormat = 'd/m/Y H:i:s e';
	private $propelLog;

	private $errorEmailSubject = "Error Build";
	private $successEmailSubject = "Success Build";

	private $buildHasError;
	private $logData;

	private $env;

	private $alreadyBackedUp;

	private $alreadyUpToDate;
	private $willNotBeUpdated;

	private $updatedVersion;
	private $currentVersion;

	function __construct() {

		$this->say("Robo task process start");

   		$this->loadConfig();
   		$this->loadAbsolutePaths();
   		$this->getCurrentVersion();

		// inicializo el log con un header
		$this->initLogWithHeader();

		$this->alreadyUpToDate = false;
   		$this->willNotBeUpdated = false;
   		$this->alreadyBackedUp = false;

   		$this->log('Robo task process start');
	}

	/**
	 * Comando que va se va a ejecutar automaticamente mediante CI
	 * @return [type] [description]
	 */
	function init() {

		$this->updateFilesWithGit();

		if (!$this->buildHasError) {

			if ($this->getRoboActions()) {

				if ($this->actions['makeBackup'] && !$this->buildHasError) {
					$this->makeCompleteBackup();
				}

				if ($this->actions['buildPropel'] && !$this->buildHasError) {
					$buildPropelResult = $this->buildPropel();
					if (!$buildPropelResult) {
						$this->restoreBackup();
					}
				}

				if ($this->actions['executeSql'] && !$this->buildHasError)
					$this->executeSqlScript();

				if (!$this->buildHasError && $this->currentVersion != $this->updatedVersion)  {
					$this->log("This app was updated to version: $this->updatedVersion");

					// actualizo el archivo currentVersion
					$this->updateLocalVersion();
				} else {
					$this->alreadyUpToDate = true;
				}

			} else {
				$this->log('Get robo actions task finished. Status: WARNING.');
				$this->log('Please verify that robo.json exists.');
			}

		} else {
			// TODO: ver de que forma obtener el error de git
			$this->log('Update with Git task Error');
		}

		$this->log("Robo task process finished");

		$this->makeReport();

		$this->say("Robo task process finish");
	}


	/**
	 * Comando que se va a ejecutar automaticamente mendiante CRONJOB
	 * @param array $opts opciones de ejecucion
	 * @param $opts[files] ejecuta el task de actualizacion de archivos
	 * @param $opts[backup] ejecuta el task de respaldo
	 * @param $opts[propel] ejecuta el build de propel
	 * @param $opts[sql] ejecuta el task de script sql
	 * @param $opts[debug] muestra en pantalla el outpu de cada task
	 */
	function runInit($opts = ['files' => false , 'backup' => false, 'propel' => false, 'sql' => false, 'debug' => false, 'backupdb' => false]) {

		$this->arrayConverter = new assoc_array2xml();

   		$runProcess = false;

		$this->debugMode = $opts['debug'];

		// en el caso de que llegue alguna opcion en el comando, no compruebo la actualizacion.
		if ($opts['files'] || $opts['backup'] || $opts['propel'] || $opts['sql'] || $opts['backupdb']) {
   			$this->actions['makeBackup'] = $opts['backup'];
   			$this->actions['updateFiles'] = $opts['files'];
   			$this->actions['executeSql'] = $opts['sql'];
   			$this->actions['buildPropel'] = $opts['propel'];
   			$this->actions['makeDbBackup'] = $opts['backupdb'];

   			$this->log('Robo was executed manually');

   			$runProcess = true;
   			$this->updatedVersion = $this->currentVersion;
   		} else
   			$runProcess = $this->verifyServerList();

   		// comienza el proceso
		if ($runProcess) {

			if ($this->actions['makeBackup'] && !$this->buildHasError)
				$this->makeCompleteBackup();

			if ($this->actions['makeDbBackup'] && !$this->buildHasError)
				$this->makeDBBackup();

			if ($this->actions['updateFiles'] && !$this->buildHasError)
				$this->updateFilesWithGit();

			if ($this->actions['buildPropel'] && !$this->buildHasError) {
				$buildPropelResult = $this->buildPropel();
				if (!$buildPropelResult)
					$this->restoreBackup();
			}

			if ($this->actions['executeSql'] && !$this->buildHasError)
				$this->executeSqlScript();

			if (!$this->buildHasError && $this->currentVersion != $this->updatedVersion) {
				$this->log("This app was updated to version: $this->updatedVersion");

				// actualizo el archivo currentVersion
				$this->updateLocalVersion();
			}

		} else {

			if ($this->alreadyUpToDate) {
				$this->log('This app is already up to date.');
			} else {
				$this->log('This app will not be updated.');
				$this->willNotBeUpdated = true;
			}
		}

		$this->log("Robo task process finished.");
		$this->makeReport();
	}

	private function updateLocalVersion() {
		$versionJson = array( 'version' => "$this->updatedVersion" );
		$this->taskWriteToFile($this->deployPath.'/version.json')
				->line(json_encode($versionJson))
				->run();
	}

 	/**
	* Carga la configuracion de la base de datos y del modulo task
	* TODO : Separar esta logica en 2 partes.
	*/
	private function loadConfig() {

		global $appDir;

		$appDir = realpath(__DIR__ . '/..');

		require_once("$appDir/config/load_config.php");

		include_once('../config/application-conf.php');

		// soporte para multiples entornos de propel
		if (!isset($_SERVER['env']))
		    $_SERVER['env'] = 'prod';
		$this->env = $_SERVER['env'];

		if (isset($conf['datasources']['application']))
			$propelDatasouce = $conf["datasources"]["application"]["connection"];
		else
			$propelDatasouce = $conf["datasources"][$this->env]["connection"];

		$dsnParts = explode("=", $propelDatasouce["dsn"]);
		$database = $dsnParts[2];
		$dsnParts2 = explode(";", $dsnParts[1]);
		$host = $dsnParts2[0];
		$user = $propelDatasouce["user"];
		$password = $propelDatasouce["password"];

		// el nombre de la aplicacion se define por el usuario que corre el script
		$this->appData['name'] = trim(shell_exec('whoami'), " \n\r");

		$this->appData['dbname'] = $database;
		$this->appData['dbuser'] = $user;
		$this->appData['dbpassword'] = $password;
		$this->appData['remoteProtocol'] = ConfigModule::get('task', 'remoteProtocol');
		$this->appData['remoteSshPort'] = ConfigModule::get('task', 'remoteSshPort');
		$this->appData['remoteUrl'] = ConfigModule::get('task', 'remoteUrl');
		$this->appData['remoteDir'] = ConfigModule::get('task', 'remoteDir');
		$this->appData['remoteFile'] = ConfigModule::get('task', 'remoteFile');
		$this->appData['remoteUsername'] = ConfigModule::get('task', 'remoteUsername');
		$this->appData['propelVersion'] = ConfigModule::get('task', 'propelVersion');

		$this->appData['gitRemote'] = ConfigModule::get('task', 'gitRemote');
		$this->appData['gitBranch'] = ConfigModule::get('task', 'gitBranch');
		$this->appData['gitUrl'] = ConfigModule::get('task', 'gitUrl');

	}

 	/**
	 * Configura los paths abosolutos que se van a utilizar en los tasks
	 */
	private function loadAbsolutePaths() {
		$this->rootPath = realpath(__DIR__ . '/..');
		$this->deployPath = realpath(__DIR__ . '/..') . '/deploy';
		$this->backupPath = realpath(__DIR__ . '/..') . '/backups';
		$this->propelBuildPath = realpath(__DIR__) . '/propel';
	}

	/**
	 * Genera respaldo completo, tanto de archivos como de base de datos
	 * @return bool $result true si se genero correctamente, false si no
	 */
	private function makeCompleteBackup() {
		$this->log('Complete backup task start');

		$filesBackup = $this->makeFilesBackup();

		if (!$filesBackup)
			return $filesBackup;

		return $this->makeDbBackup();
	}

	/**
	 * Genera respaldo de base de datos. dump+insert
	 * @return boolean 	Resultado de la operacion
	 */
	private function makeDbBackup() {
		$this->log('Database backup task start');

		$this->_mkdir("$this->backupPath/files");

		$result = $this->taskExec("sh $this->backupPath/dbBackup.sh")
					->args($this->appData['name'])
					->args($this->appData['dbname'])
					->args($this->appData['dbuser'])
					->args($this->appData['dbpassword'])
					->args($this->rootPath)
					->run();

		if ($result->wasSuccessful()) {
			$this->log('Database backup task finish. Status: OK');
			$this->alreadyBackedUp = true;
			return true;
		}
		else {
			$this->log('Database backup task finish. Status: ERROR. Message: '.$result->getMessage());
			$this->buildHasError = true;
			return false;
		}
	}

	/**
	 * Genera respaldo de archivos, omitiendo los directorios de git, backup y templates_c
	 * @return boolean Resultado de la operacion
	 */
	private function makeFilesBackup() {
		$this->log('Files backup task start');

		$this->_mkdir("$this->backupPath/files");

		$result = $this->taskExec("sh $this->backupPath/filesBackup.sh")
					->args($this->appData['name'])
					->args($this->appData['dbname'])
					->args($this->appData['dbuser'])
					->args($this->appData['dbpassword'])
					->args($this->rootPath)
					->run();

		if ($result->wasSuccessful()) {
			$this->log('Files backup task finish. Status: OK');
			return true;
		}
		else {
			$this->log('Files backup task finish. Status: ERROR. Message: '.$result->getMessage());
			$this->buildHasError = true;
			return false;
		}
	}

    /**
	 * Restaura un respaldo del sistema y de la base de datos. Elimina los templates
	 * y los archivos .data que existan dentro del directorio config
	 * @return bool $result true si se restauro correctamente, false si no
	 */
	private function restoreBackup() {

		$this->log('Restore backup task start');

		// obtengo el ultimo archivo creado
		$fileToRestore = shell_exec("ls -Art $this->backupPath/files | tail -n 1");

		if (!empty($fileToRestore)) {

			$result = $this->taskExec("sh $this->backupPath/restoreBackup.sh")
					->args($this->appData['name'])
					->args($fileToRestore)
					->args($this->rootPath)
					->printed($this->debugMode)
					->run();

			if ($result->wasSuccessful()) {

				$cleanSmartTemplatesResult = $this->taskCleanDir(['smarty_tpl/templates_c'])->run();

				if ($cleanSmartTemplatesResult->wasSuccessful())
					$this->log('Smarty templates_c directory cleaned.');
				else
					$this->log($cleanSmartTemplatesResult->getMessage());

				$deleteDataConfigFiles = $this->taskExec("find $this->rootPath/config -name '*.data' -type f -delete")
																		->run();

				$deleteDataPhpmvcFiles = $this->taskExec("find $this->rootPath/WEB-INF -name '*.data' -type f -delete")
																		->run();

				if ($deleteDataConfigFiles->wasSuccessful())
					$this->log('Data config files removed.');
				else
					$this->log($deleteDataConfigFiles->getMessage());

				if ($deleteDataPhpmvcFiles->wasSuccessful())
					$this->log('Data phpmvc files removed.');
				else
					$this->log($deleteDataPhpmvcFiles->getMessage());

				$this->log('Restore backup task finished. Status: OK. File restored: '.$fileToRestore);
				return true;
			} else {
				$this->log('Restore backup task finished. Status: ERROR. Message: '.$result->getMessage());
				$this->buildHasError = true;
				return false;
			}
		} else {
			$this->log('No backup file available to restore.');
			return false;
		}

	}

	/**
	 * Ejecuto un script sql, definido en el servidor de actualizacion
	 * @return bool $result true si se ejecuto correctamente, false si no
	 */
	private function executeSqlScript() {

		$logFileName = $this->deployPath.'/logs/execute-sql.log';
		$this->_touch($logFileName);

		$backupResult = true;

		// al ser un proceso critico, siempre que se ejecute esta task, tiene que crearser o haberse creado un backup
		if (!$this->actions['makeBackup'] && !$this->alreadyBackedUp)
			$backupResult = $this->makeDbBackup();

		if ($backupResult) {

			$this->log('Execute SQL task started');

			// armo el script scp con los datos del servidor de actualizacion
			$scpFilePath = "-P " . $this->appData['remoteSshPort'] . " " . $this->appData['remoteUsername'] . '@' . $this->appData['remoteUrl'] . ':' . $this->appData['remoteDir'] . $this->sqlBuildScriptFile;

			$copySqlScriptResult = $this->taskExec("scp -r $scpFilePath $this->deployPath/scripts/buildScript.sql 2> $logFileName")->run();

			if ($copySqlScriptResult->wasSuccessful()) {

				$dbuser = $this->appData['dbuser'];
				$dbpass = $this->appData['dbpassword'];
				$dbname = $this->appData['dbname'];

				// ejecucion del script sql por mysql ctl
				$result = $this->taskExec("mysql -u $dbuser -p$dbpass $dbname < $this->deployPath/scripts/buildScript.sql 2> $logFileName")->run();

				if ($result->wasSuccessful()) {
					$this->log('Execute SQL task finished. Status: OK');
					return true;
				} else {
					$this->log('Execute SQL task finished with ERROR. Message: '.file_get_contents($logFileName));
					$this->buildHasError = true;
					return false;
				}
			} else {
				$this->log('Download SQL script task finshed with ERROR. Message: '.file_get_contents($logFileName));
				$this->log('WARNING: SQL script was not executed. Robo task will continue now.');
				return true;
			}

			$this->_remove("$this->deployPath/scripts/buildScript.sql");
		}

		$this->_remove($logFileName);
	}

	/**
	 * Ejecuta los scripts de propel para aplicar los cambios a la base de datos local
	 * y regenerar las clases
	 * @return bool $result true si se genero correctamente, false si no
	 */
	private function buildPropel() {

		$this->log('Propel build started');

		// Objeto que conoce los errores comunes de propel
		$this->errorParser = new PropelErrorParser();

		$this->propelLog = $this->deployPath.'/logs/propel-build.log';
		$this->_touch($this->propelLog);

		$backupResult = true;

		// al ser un proceso critico, siempre que se ejecute esta task, tiene que crearser o haberse creado un backup
		if (!$this->actions['makeBackup'] && !$this->alreadyBackedUp)
			$backupResult = $this->makeDbBackup();

		if ($backupResult) {

			if ($this->runPropelGen()) {
				if ($this->runPropelDiff()) {
					if ($this->runPropelMigrate()) {
						$this->log('Propel build finished. Status: OK');
						return true;
					}
					else{
						$this->buildHasError = true;
						return false;
					}
				}
				else{
					$this->buildHasError = true;
					return false;
				}

			} else{
				$this->buildHasError = true;
				return false;
			}
		}

		// elimino el log de error de propel
		$this->_remove($this->propelLog);
	}

	/**
	 * Generar clases de propel
	 * @return bool $result true si se genero correctamente, false si no
	 */
	private function runPropelGen() {

		$propelVersion = $this->appData['propelVersion'];

		// el path del archivo tiene que tomarse desde el dir que se le pasa como metodo al taskExec
		if (empty($propelVersion)) {
			$command = "sh WEB-INF/vendor/bin/console propel:build 2> $this->propelLog";
			$executeFrom = $this->rootPath;
		} else {
			$command = "sh ../lib-phpmvc/Propel/$propelVersion/generator/bin/propel-gen 2> $this->propelLog";
			$executeFrom = 'propel/';
		}

		$this->taskExec($command)
			->dir($executeFrom)
			->printed($this->debugMode)
			->run();

		// busco si el comando de propel genero algun error
		$errorFinding = $this->taskExec("grep 'Aborting'")
					->args($this->propelLog)
					->dir($this->rootPath.'/WEB-INF/')
					->run();

		if ($errorFinding->wasSuccessful()){
			// parse el error y lo escribo en el log que se envia por mail
			$errorData = $this->errorParser->parse(file_get_contents($this->propelLog));
			$this->log('Propel gen command finished. Status: ERROR. Message: '.$errorData);
			return false;
		}
		else {
			$this->log('Propel gen command finished. Status: OK');
			return true;
		}
	}

	/**
	 * Generar sql de diferencias de propel con base exitente
	 * @return bool $result true si se genero correctamente, false si no
	 */
	private function runPropelDiff() {

		$propelVersion = $this->appData['propelVersion'];
		$env = $this->env;

		//elimino todas las migrations existentes, el diff va a generar nuevas.
		$this->taskCleanDir(['propel/migrations'])->run();

		// el path del archivo tiene que tomarse desde el dir que se le pasa.
		if (empty($propelVersion)) {
			$command = "sh WEB-INF/vendor/bin/console propel:diff $env 2> $this->propelLog";
			$executeFrom = $this->rootPath;
		} else {
			$command = "sh ../lib-phpmvc/Propel/$propelVersion/generator/bin/propel-gen diff 2> $this->propelLog";
			$executeFrom = 'propel/';
		}

		$this->taskExec($command)
				->dir($executeFrom)
				->printed($this->debugMode)
				->run();

		$errorFinding = $this->taskExec("grep 'Aborting'")
					->args($this->propelLog)
					->dir($this->rootPath.'/WEB-INF/')
					->run();

		if ($errorFinding->wasSuccessful()){
			// parse el error y lo escribo en el log que se envia por mail
			$errorData = $this->errorParser->parse(file_get_contents($this->propelLog));
			$this->log('Propel diff command finished. Status: ERROR. Message: '.$errorData);
			return false;
		}
		else {
			$this->log('Propel diff command finished. Status: OK');
			return true;
		}
	}

	/**
	 * Ejecuta la migracion de propel
	 * @return bool $result true si se ejecuto correctamente, false si no
	 */
	private function runPropelMigrate() {

		$propelVersion = $this->appData['propelVersion'];
		$env = $this->env;

		// el path del archivo tiene que tomarse desde el dir que se le pasa.
		if (empty($propelVersion)) {
			$command = "sh WEB-INF/vendor/bin/console propel:migrate $env 2> $this->propelLog";
			$executeFrom = $this->rootPath;
		} else {
			$command = "sh ../lib-phpmvc/Propel/$propelVersion/generator/bin/propel-gen migrate 2> $this->propelLog";
			$executeFrom = 'propel/';
		}

		$this->taskExec($command)
			->dir($executeFrom)
			->printed($this->debugMode)
			->run();

		$errorFinding = $this->taskExec("grep 'Aborting'")
					->args($this->propelLog)
					->dir($this->rootPath.'/WEB-INF/')
					->run();

		if ($errorFinding->wasSuccessful()){
			// parse el error y lo escribo en el log que se envia por mail
			$errorData = $this->errorParser->parse(file_get_contents($this->propelLog));
			$this->log('Propel migrate command finished. Status: ERROR. Message: '.$errorData);
			return false;
		}
		else {
			$this->log('Propel migrate command finished. Status: OK');
			return true;
		}
	}

 	/**
	 * Revisa si hay cambios en los schemas de propel y configura la variable $this->hasSchemaChanged
	 *
	 * NO SE UTILIZA
	 */
	private function watchSchemaChange() {
		$this->taskWatch()
			->monitor('propel', function() {
				$this->hasSchemaChanged = true;
			})->background()->run();
	}

 	/**
	 * Controla los archivos de actualizacion y define los task que se van a ejecutar
	 * @return bool $result true si se ejecuto correctamente, false si no
	 */
	private function verifyServerList() {

		$this->log('Get updated server list task start');

		$url = $this->appData['remoteProtocol'] . "://" . $this->appData['remoteUrl'] . "/admin/Main.php?do=tablerosApiGetUpdateData&server=" . $this->appData['name'];

		$serverXmlData = simplexml_load_file($url);

		$this->say($serverXmlData->tablero->attributes()->version);

		// chequeo si la version es mayor a la actual
		if  (version_compare($serverXmlData->tablero->attributes()->version, $this->currentVersion)) {

			$this->actions['makeBackup'] = !empty($serverXmlData->tablero->attributes()->makeBackup) ? $serverXmlData->tablero->attributes()->makeBackup : false;
			$this->actions['updateFiles'] = !empty($serverXmlData->tablero->attributes()->updateFiles) ? $serverXmlData->tablero->attributes()->updateFiles : false;
			$this->actions['buildPropel'] = !empty($serverXmlData->tablero->attributes()->buildPropel) ? $serverXmlData->tablero->attributes()->buildPropel : false;
			$this->actions['executeSql'] = !empty($serverXmlData->tablero->attributes()->executeSql) ? $serverXmlData->tablero->attributes()->executeSql : false;

			$this->updatedVersion = $serverXmlData->tablero->attributes()->version;
			$this->say('Update to version: '. $this->updatedVersion);
			$this->log('Get updated server list task finish');
			return true;

		} else {
			$this->alreadyUpToDate = true;
			return false;
		}

	}

	/**
	 * Inicializa el log de acciones realizadas con titulo definido
	 */
	private function initLogWithHeader() {
		$this->log = "-------------------------------------------------------------------------------------------------------------------<br>" .
					 "Robo task start @ " . $this->appData['name'] . " - " . date($this->timestampFormat) . "<br>" .
					 "Current version: ".$this->currentVersion . "<br>" .
					 "-------------------------------------------------------------------------------------------------------------------<br>";
	}

	/**
	 * Genera el log de acciones realizadas apartir del comando recibido
	 * @param string $command comando solicitado
	 */
	private function log($command) {
		$this->log = $this->log . "[ " . date('H:i:s') . " ] - " . $command . "<br>";
	}


	/**
	 * Genera un reporte por email o archivo con el log de ejecucion
	 */
	private function makeReport() {
		if ( !$this->writeReportToEmail() ) {
			$this->writeReportToFile();
		}
	}

	/**
	 * Genera un archivo con el resultado de la ejecucion del robo.
	 * @return boolean Resultado de la ejecucion.
	 */
	private function writeReportToFile() {

		if (!file_exists($this->deployPath.'/logs/robo.log'))
			$this->_touch($this->deployPath.'/logs/robo.log');

		$writeReportFileTask = $this->taskWriteToFile($this->deployPath.'/logs/robo.log')
									->text($this->log)
									->run();

		return $writeReportFileTask->wasSuccessful();
	}

	/**
	 * Genera un mensaje de correo electronico con el resultado de la ejecucion del robo
	 * @return boolean $result true si se envio false si no
	 */
	private function writeReportToEmail() {

		global $system;

		try {

			$manager = new EmailManagement();

			// obtengo los datos del smtp configurado localmente
			$appHostname = $system["config"]["email"]["smtpOptions"]["smtpHost"];
	 		$fromEmail = $system["config"]["system"]["parameters"]["debugMail"];
	 		$toEmail = $system["config"]["system"]["parameters"]["debugMail"];

			if ($this->buildHasError)
				$subject = "Robo Report - FAILED @ ";
			else {
				if ($this->alreadyUpToDate)
					$subject = "Robo Report - ALREADY UP TO DATE @ ";
				else if ($this->willNotBeUpdated)
					$subject = "Robo Report - NO ACTION @ ";
				else
					$subject = "Robo Report - SUCCESS @ ";
			}

			$subject .= $appHostname;

			$message = $manager->createHTMLMessage($subject, $this->log);

			return $manager->sendMessage($toEmail, $fromEmail, $message);

		} catch (Exception $e) {
			return false;
		}

	}

	/**
	 * Metodo publico para actualizar los archivos de configuracion
	 * @return [type] [description]
	 */
	function runConfigUpdate() {
		$this->updateConfig();
	}

	/**
	 * Actualiza el config.local con la diferencia datos entre config.xml + config.local.xml y un archivo de configuracion base
	 * @return [type] [description]
	 */
	private function updateConfig() {

		// hacer un diff con el config que ya esta generado en system
		$xmlManager = new ConfigXmlManagement();

		global $system;

		$localConfigArray = $system['config'];

		$configBaseDir = $this->appData['remoteProtocol'] . "://" . $this->appData['remoteUrl'] . "/" . "config_base.xml";

		$file = file_get_contents($configBaseDir);

		if (!$file) {
			$this->say('Verifique que el archivo de configuracion base exista en '. $configBaseDir);
		} else {

			$baseConfig = simplexml_load_string($file , "SimpleXMLElement", LIBXML_NOCDATA);
			$baseConfigJson = json_encode($baseConfig);

			$baseConfigArray = json_decode($baseConfigJson, TRUE);

			$diff = $xmlManager->diff($localConfigArray, $baseConfigArray);

			$newXml = new SimpleXMLElement('<?xml version="1.0" encoding="UTF-8"?> <config/>', null, false);
			$xmlManager->toXml($newXml, $diff);

			$domxml = new DOMDocument('1.0');
			$domxml->formatOutput = true;
			$domxml->loadXML($newXml->asXML());
			$domxml->save(realpath(__DIR__ . '/..') .'/config/config.local.xml');

			$this->say('Archivo de configuracion local actualizado.');
		}
	}

	private function getRoboActions() {

		$this->log('Get robo actions task started.');

		if (!file_exists($this->deployPath.'/robo.json')) {
			return false;
		}

		$roboJson = json_decode(file_get_contents($this->deployPath.'/robo.json'), true);

		$this->updatedVersion = $roboJson['updatedVersion'];

		$this->actions['makeBackup'] = $roboJson['actions']['backup'];
		$this->actions['buildPropel'] = $roboJson['actions']['propel'];
		$this->actions['executeSql'] = $roboJson['actions']['sql'];

		$this->log('Get robo actions task finished.');

		return true;
	}

	/**
	 * Metodo publico para incializar repositorio git
	 * @return [type] [description]
	 */
	function runGitInitRepo() {
		$this->initGitRepository();
	}

	/**
	 * Inicializa repositorio git y lo empareja con el configurado
	 * @return [type] [description]
	 */
	private function initGitRepository() {
		$logFileName = $this->deployPath.'git-init.log';
		$this->_touch($logFileName);

		$repoTmpDir = "$this->rootPath/tmp-git-dir.tmp";

		$this->_mkdir($repoTmpDir);

		$this->taskGitStack()
			->cloneRepo($this->appData['gitUrl'], $repoTmpDir)
			->run();

		$copyGit = $this->taskExec("mv $repoTmpDir/.git $this->rootPath 2> $logFileName")->run();

		if ($copyGit->wasSuccessful()) {
			$rm = $this->taskExec("rm -rf $repoTmpDir")->run();

			if ($rm->wasSuccessful()) {
				$this->taskExec("git reset --hard HEAD")
								->dir($this->rootPath)
								->run();
			} else {
				$this->buildHasError = true;
				return false;
			}

		} else {
			$this->buildHasError = true;
			return false;
		}
	}

	// Update files methods

	/**
	 * Metodo publico para actualizar los archivos mediante scp desde servidor de actualizacion
	 * @return [type] [description]
	 */
	function runScpUpdate() {
		$this->updateFilesWithScp();
	}

	/**
	 * Metodo publico para actualizar los archivos mediante git
	 * @return [type] [description]
	 */
	function runGitPull() {
		$this->updateFilesWithGit();
	}

	/**
	 * Copia los archivos actualizados al tablero mediante scp al servidor remoto de actualizacion
	 * @return bool $result true si se copiaron correctamente, false si no
	 */
	private function updateFilesWithScp() {

		$logFileName = 'update-files-scp.log';
		$this->_touch($logFileName);

		$this->log('Updates files with SCP task started');

		// armo el path scp desde donde voy a copiar los archivos
		$updatedFilesPath = "-P " . $this->appData['remoteSshPort'] . " " . $this->appData['remoteUsername'] . '@' . $this->appData['remoteUrl'] . ':' . $this->appData['remoteDir'] . $this->updatedFilesFolder;

		// los archivos se van a copiar al raiz de la aplicacion 'public_html'
		$root = realpath(__DIR__ . '/../..');
		$result = $this->taskExec("scp -r $updatedFilesPath $root 2> $logFileName")
					->run();

		if ($result->wasSuccessful()) {
			$this->log('Update files task finished. Status: OK');
			return true;
		}
		else {
			$this->log('Update files task finished. Status: ERROR. Message: '.file_get_contents($logFileName));
			$this->buildHasError = true;
			return false;
		}

		$this->_remove($logFileName);
	}

	/**
	 * Ejecuta git pull para traer los ultimos cambios del repositorio configurado
	 * @return [type] [description]
	 */
	private function updateFilesWithGit() {
		$this->log('Update files with GIT task started.');

		$update = $this->taskGitStack()
					->pull($this->appData['gitRemote'], $this->appData['gitBranch'])
					->printed(false)
					->run();

		if ($update->wasSuccessful()) {
			$this->log('Update files with GIT task finished. Status: OK');
			return true;
		} else {
			$this->log('Update files with GIT task finished. Status: ERROR');
			$this->buildHasError = true;
			return false;
		}
	}

	/**
	 * Muestra cual es la version actual del sistema
	 * @return string Version actual
	 */
	public function getCurrentVersion() {

		if (!file_exists($this->deployPath.'/version.json')) {
			$this->_touch($this->deployPath.'/version.json');
			$this->currentVersion = 0;
			$this->say('Current version is not set yet');
			return true;
		}

		$currentVersionJson = json_decode(file_get_contents($this->deployPath.'/version.json'), true);

		if ($currentVersionJson) {
			$this->currentVersion = $currentVersionJson['version'];
			$this->say('Current version is: '.$this->currentVersion);
		}
	}
}
