<?php

/**
 * BaseAction
 * Clase base de acciones a ejecutar por el modelo
 *
 * @package phpMVC
 * @author Modulos Empresarios / Egytca
 * @copyright Copyright (c) 2015, Egytca
 */

use Egytca\DebugData;
use Egytca\ErrorReporting\GlobalErrors;
use Egytca\ErrorReporting\GlobalWarnings;
use Egytca\GlobalApp;
use Egytca\Utils;

include_once('Include.inc.php');
include_once('TimezonePeer.php');
include_once('Common.class.php');
include_once('Action.php');
require_once('Smarty_config.inc.php');

require_once('BaseQuery.php');
require_once('BaseListAction.php');
require_once('BaseDoEditAction.php');
require_once('BaseDoDeleteAction.php');
require_once('BaseSelectAction.php');

/**
 * Acciones basicas a ejecutar por el modelo, desde comprobacion de permisos hasta redireccionamiento de salida.
 */
class BaseAction extends Action {

	protected $app;
	protected $debugData;
	protected $egytca;
	protected $errors;
	protected $smarty;
	protected $warnings;
	protected $actionRequested;
	protected $forwardName = 'success';
	protected $forwardFailureName = 'failure';

 /**
	* Constructor
	*/
	function __construct() {
		$this->egytca = [];
	}

 /**
  * Procesa la solicitud HTTP y genera la correspondiente respuesta HTTP
  * Devuelve una instancia del ActionForward describiendo donde y como se debe 
  * direccionar la respuesta o NULL si la respuesta ya se completo
	*
	* @param $mapping    El ActionConfig usado para esta instancia
	* @param $form       El ActionForm opcional
	* @param $request    La solicitud HTTP (HttpRequestBase) que se esta procesando
	* @param $response   La respuesta HTTP (HttpRequestBase) que se esta generando
	* @returns ActionForward  Redireccionamiento o salida de la ejecucion
	*/

	function execute($mapping, $form, &$request, &$response) {

		parent::execute($mapping, $form, $request, $response);

		$plugInKey = 'SMARTY_PLUGIN';
		$smarty =& $this->actionServer->getPlugIn($plugInKey);
		if ($smarty == NULL)
			echo 'No PlugIn found matching key: ' . $plugInKey . '<br>\n';

		$this->smarty = $smarty;

		$this->smarty->assign('app', GlobalApp::getObject());
//		$this->smarty->assign('debugTool', false);
		$this->setupDebugData();
		$this->egytca['angularDateFormat'] = Utils::getAngularDateFormat();
		$this->egytca['routeConfig'] = json_encode(include 'config/routes.php');
		$this->smarty->assign('egytca', $this->egytca);

		$this->setModuleConfig();

		setlocale(LC_ALL, Common::getCurrentLocale());
		$GLOBALS['_NG_LANGUAGE_'] =& $smarty->language;

		//Tiene Prioriodad si se pasa el lenguage en el request.
		//Analizar si es necesario agregar luego control de GoogleBot
		if (!empty($_REQUEST['lang'])) {
			$GLOBALS['_NG_LANGUAGE_']->setCurrentLanguage($_REQUEST['lang']);
			Common::setCurrentLanguageCode($_REQUEST['lang']);
		}
		else {
			if (!empty($GLOBALS['_NG_LANGUAGE_']))
				$GLOBALS['_NG_LANGUAGE_']->setCurrentLanguage(Common::getCurrentLanguageCode());
		}

		$this->actionRequested = $request->getAttribute('ACTION_DO_PATH');

		$this->smarty->assign('configModule', new ConfigModule());
		$this->smarty->assign('actualAction', $this->actionRequested);

		$_SERVER['FULL_URL'] = 'http';
		if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] == 'on')
				$_SERVER['FULL_URL'] .=  's';
		$_SERVER['FULL_URL'] .=  '://';
		$serverPort = '';
		if ($_SERVER['SERVER_PORT']!='80')
			$serverPort = ':' . $_SERVER['SERVER_PORT'];

		$systemUrl = $_SERVER['FULL_URL'] . $_SERVER['HTTP_HOST'] . substr($_SERVER['REQUEST_URI'], 0, strrpos($_SERVER['REQUEST_URI'], '/')) . $serverPort . '/Main.php';
		$this->smarty->assign('systemUrl', $systemUrl);
		$scriptPath = substr($_SERVER['PHP_SELF'], 0, (strlen($_SERVER['PHP_SELF']) -8 - @strlen(!isset($_SERVER['PATH_INFO']) ? : $_SERVER['PATH_INFO'])));
		$this->smarty->assign('scriptPath', $scriptPath);
		//Chequeo si esta en mantenimiento
		if (Common::inMaintenance()) {
			header('Location: Main.php?do=commonMaintenance');
			exit;
		}

		// Inicio verificaciones de seguridad
		global $loginPath;

		// Chequeo si esta bloqueada la ip desde donde se hace el request
		if((Common::checkBlockedIp() || Common::checkLoginIpFailures()) && $this->actionRequested != 'securityBlocked') {// La ip de donde se conecta esta bloquada
			header('Location:Main.php?do=securityBlocked');
			exit();
		}

		$securityAction = SecurityActionQuery::create()->filterByNameOrPair($this->actionRequested)->findOne();
		$securityModule = SecurityModuleQuery::create()->filterByModule($this->module)->findOne();

		// Controlo las acciones y modulos que no requieren login
		// Si no se requiere login, $noCheckLogin va a ser igual a 1
		$noCheckLogin = 1;
		if (!empty($securityAction)) // El action, si esta presente, predomina sobre el modulo
			$noCheckLogin = $securityAction->getOverallNoCheckLogin();
		else if (!empty($securityModule)) // Si no esta el action, se busca en el modulo
			$noCheckLogin = $securityModule->getNoCheckLogin();
		else
			$noCheckLogin = 0;

		if (ConfigModule::get('global', 'noCheckLogin'))   // Si se configura el sistema sin seguridad, predomina sobre lo anterior
			$noCheckLogin = 1;

		// Inicio de la salida
		header('Content-type: text/html; charset=UTF-8');

		$loggedUser = Common::getLoggedUser();

		if (!$noCheckLogin && $this->actionRequested != 'securityBlocked') { // Verifica login $noCheckLogin != 1
			if (!empty($loggedUser)) {
				//Veo que el usuario no este bloqueado
				if(Common::isBlockedUser($loggedUser->getUsername()) || Common::checkLoginUserFailures($loggedUser)) {// Se bloqueo usuario
					Common::blockedUserUnsetSession();
					header('Location:Main.php?do=' . $loginPath . '&message=blockedUser');
					exit();
				}
				Common::setLastActionTime($loggedUser);
				if (!ConfigModule::get('global', 'noSecurity') && $this->actionRequested != 'securityNoPermission') { // Si seguridad esta habilitada y verifica permisos
					if (!empty($securityAction))
						$access = $securityAction->getAccessByUser($loggedUser);
					else if (!empty($securityModule))
						$access = $securityModule->getAccessByUser($loggedUser);

					if (empty($access)) {// No tiene permiso
						header('Location:Main.php?do=securityNoPermission');
						exit();
					}
				}
				else {
					//No verifica seguridad
				}
			}
			else { //Si requiere login y no hay sesion va a login
				if ($this->actionRequested != $loginPath && $this->actionRequested != 'commonDoLogin' && $this->actionRequested != 'commonLogin' && $this->actionRequested != 'usersLogin' &&
							$this->actionRequested != 'usersDoLogin' && $this->actionRequested != 'securityBlocked') { // Si las Acciones difieren a las que no requieren login
					$_SESSION['loginRequestReferrer'] = $_SERVER['QUERY_STRING'];   // Guardo el query para redireccionar luego del login
					header('Location:Main.php?do=' . $loginPath);
					exit();
				}
			}
		}
		else {
			// No verifica login
		}

		// Asigno objeto usuario logueado a smarty
		if (isset($_SESSION['loginUser']) && is_object($_SESSION['loginUser']) && get_class($_SESSION['loginUser']) == 'User')
			$this->smarty->assign('loginUser', $_SESSION['loginUser']);
		else if (isset($_SESSION['loginAffiliateUser']) && is_object($_SESSION['loginAffiliateUser']) && get_class($_SESSION['loginAffiliateUser']) == 'AffiliateUser')
			$this->smarty->assign('loginAffiliateUser', $_SESSION['loginAffiliateUser']);
		else if (isset($_SESSION['loginClientUser']) && is_object($_SESSION['loginClientUser']) && get_class($_SESSION['loginClientUser']) == 'ClientUser')
			$this->smarty->assign('loginClientUser', $_SESSION['loginClientUser']);
		else if (isset($_SESSION['loginRegistrationUser']) && is_object($_SESSION['loginRegistrationUser']) && get_class($_SESSION['loginRegistrationUser']) == 'RegistrationUser')
			$this->smarty->assign('loginRegistrationUser', $_SESSION['loginRegistrationUser']);


		$this->smarty->assign('currentLanguageCode', Common::getCurrentLanguageCode());

		$this->smarty->assign('browser', Common::getBrowser());
		$this->smarty->assign('isBot', Common::isBot());

		// TODO Se requiere para los include de modulos, revisar para ver si se puede eliminar el assign
		$this->smarty->assign('mapping', $mapping);

		$this->template = new SmartyOutputFilter();
		if (defined('Smarty::SMARTY_VERSION'))
			$smarty->registerFilter('output', array($this->template, 'smarty_add_template'));
		else
			$smarty->register_outputfilter(array($this->template, 'smarty_add_template'));

		if ($this->isAjax()) {
			$this->template->template = 'TemplateAjax.tpl';
			$this->smarty->assign('isAjax', true);
		}

		$systemParameters = Common::getModuleConfiguration('system');
		$this->smarty->assign('parameters', $systemParameters['parameters']);
		$this->setPlanningYears();
		$this->smarty->assign('SESSION', $_SESSION);

		if (!empty($GLOBALS['_NG_LANGUAGE_'])) {
			if (defined('Smarty::SMARTY_VERSION'))
				$smarty->registerFilter('output', 'smarty_outputfilter_i18n');
			else
				$smarty->register_outputfilter('smarty_outputfilter_i18n');
		}

		$this->smarty->assign('languagesAvailable', Common::getAllLanguages());		

		// Ejecucion con forward por defecto
		if ($this->exec() === false)
			return $mapping->findForwardConfig($this->forwardFailureName);
		else
			return $mapping->findForwardConfig($this->forwardName);

	} //End execute

	/**
	 * Ejecucion por defecto para porceder a display y forward
	 */
	protected function exec() {

		// Informacion para armar los links de paginador
		$this->assignFiltersToSmarty();

	}
	
	/**
	 * Agrega la informacion para el Debug
	 */
	private function setupDebugData() {
		$this->debugData = new DebugData();
		$this->errors = GlobalErrors::getObject();
		$this->warnings = GlobalWarnings::getObject();
		$this->smarty->assign('debugData', $this->debugData);
		$this->debugData->set('errors', $this->errors);
		$this->debugData->set('warnings', $this->warnings);
	}

	private function setPlanningYears() {
		$thisYear = date('Y');
		if (!isset($_SESSION['planning']['startingYear']))
			$_SESSION['planning']['startingYear'] = $thisYear;
		if (!isset($_SESSION['planning']['endingYear']))
			$_SESSION['planning']['endingYear'] = $thisYear;
	}

	/**
	 * Agrega a smarty la configuracion del modulo en base a la accion
	 */
	protected function setModuleConfig() {
		if (!isset($this->module))
			$this->module = Common::getModuleFromActionName(get_class($this));
		$this->smarty->assign('module', $this->module);
		$this->smarty->assign('moduleConfig', Common::getModuleConfiguration($this->module));
	}

	/**
	 * Prepara los valores de filtrado y pagina para utilizar en paginadores
	 */
	protected function prepareFilters() {

		if (!empty($_REQUEST['filters']))
			$this->filters = $_REQUEST['filters'];
		if (isset($_REQUEST['page']) && $_REQUEST['page'] > 0)
			$this->params['page'] = $_REQUEST['page'];
		if (!empty($this->filters['perPage']))
			$this->perPage = $this->filters['perPage'];

	}

	/**
	 * Asigna a Smarty los valores de filtros, paginas, mensajes, etc. que se pasan por defecto
	 */
	protected function assignFiltersToSmarty() {

		// Informacion para armar los links de paginador
		$url = 'Main.php?do=' . lcfirst(substr_replace(get_class($this), '', strrpos(get_class($this), 'Action'), 6));

		if (isset($this->filters))
			$url .= '&' . http_build_query(['filters' => $this->filters]);

		$this->smarty->assign('url', $url);
		$this->smarty->assign('filters', isset($_GET['filters']) ? $_GET['filters'] : NULL);
		$this->smarty->assign('page', isset($_GET['page']) ? $_GET['page'] : NULL);
		$this->smarty->assign('message', isset($_GET['message']) ? $_GET['message'] : NULL);

	}

	/**
	 * Agrega parametros al url de un forward
	 * @param $params array with parameters with key and value
	 * @param $mapping
	 * @param $forwardName nombre del forward que se quiere modificar de ese mapping
	 * @returns ForwardConfig con los parametros agregados
	 */
	function addParamsToForwards($params, $mapping, $forwardName) {

		$myRedirectConfig = $mapping->findForwardConfig($forwardName);
		$myRedirectPath = $myRedirectConfig->getpath();

		foreach ($params as $key => $value)
			$myRedirectPath .= "&$key=" . htmlentities(urlencode($value));

		return new ForwardConfig($myRedirectPath, True);

	}

	/**
	 * Agrega parametros al url de un forward
	 * @param $params array with parameters with key and value
	 * @param $mapping
	 * @param $forwardName nombre del forward que se quiere modificar de ese mapping
	 * @returns ForwardConfig con los parametros agregados
	 */
	function addFiltersToForwards($params, $mapping, $forwardName) {

		$myRedirectConfig = $mapping->findForwardConfig($forwardName);
		$myRedirectPath = $myRedirectConfig->getpath();

		foreach ($params as $key => $value)
			$myRedirectPath .= "&filters[$key]=" . htmlentities(urlencode($value));

		return new ForwardConfig($myRedirectPath, True);

	}

	/**
	 * Agrega parametros al url de un forward
	 * @param $params array with parameters with key and value
	 * @param $mapping
	 * @param $forwardName nombre del forward que se quiere modificar de ese mapping
	 * @returns ForwardConfig con los parametros agregados
	 */
	function addParamsAndFiltersToForwards($params, $filters, $mapping, $forwardName) {

		$myRedirectConfig = $mapping->findForwardConfig($forwardName);
		$myRedirectPath = $myRedirectConfig->getpath();

		foreach ($params as $key => $value)
			$myRedirectPath .= "&$key=" . htmlentities(urlencode($value));

		foreach ($filters as $key => $value)
			$myRedirectPath .= "&filters[$key]=" . htmlentities(urlencode($value));

		return new ForwardConfig($myRedirectPath, True);

	}

	/**
	 * Realiza el procesamiento de filtros sobre una clase Peer de Propel
	 * @param Class $peer instancia de clase peer de propel
	 * @param array $filterValuer valores de filtro a verificar, los metodos de set en la clase peer deben tener antepuesto a estos nombres, 'set'
	 * @param $smarty instancia de smarty sobre la cual se esta trabajando (tener en cuenta que al trabajar con una referencia a smarty, no hay problema de pasaje por parametro)
	 * @returns Peer con los filtros aplicados
	 */
	function applyFilters($peer, $filters, $smarty = '') {
		if (!empty($smarty))
			$this->smarty->assign('filters', $filters);
		foreach(array_keys($peer->filterConditions) as $filterKey)
			if (isset($filters[$filterKey])) {
				$filterMethod = $peer->filterConditions[$filterKey];
				$peer->$filterMethod($filters[$filterKey]);
			}
		return $peer;
	}

	/**
	 * Consulta la base de datos y obtiene la información básica que generalmente es requerida por una vista de formulario
	 * sencilla de una entidad. Tener en cuenta que la entidad propiamente dicha debe ser asignada por separado.
	 *
	 * En la vista quedan accesibles las instancias de entidades relacionadas con sus respectivos nombres pluralizados.
	 * Además se asigna el nombre tentativo del formulario que contiene la vista a incluir si se trata de un formulario embutido.
	 *
	 * @param $objectClassName nombre de la clase php de la entidad.
	 * @param $smarty instancia de smarty sobre la cual se esta trabajando (tener en cuenta que al trabajar con una referencia a smarty, no hay problema de pasaje por parametro)
	 */
	function prepareEmbeddedForm($objectClassName, $smarty) {
		$objectPeerName = $objectClassName . 'Peer';
		if (class_exists($objectPeerName)) {
			$tableMap = call_user_func(array($objectPeerName, 'getTableMap'));
			$objectPackageName = $tableMap->getPackage();
			$objectModuleName = ucwords(preg_replace('/.classes$/', '', $objectPackageName));
			$pluralizedObjectClassName = Common::pluralize($tableMap->getClassName());
			if ($objectModuleName != $pluralizedObjectClassName)
				$formTemplateName = $objectModuleName . $pluralizedObjectClassName . 'Form.tpl';
			else
				$formTemplateName = $pluralizedObjectClassName . 'Form.tpl';
			$this->smarty->assign('formTemplateName', $formTemplateName);
			$relations = $tableMap->getRelations();
			foreach ($relations as $relation) {
				if ($relation->getType() == RelationMap::MANY_TO_ONE) {
					$foreignTable = $relation->getForeignTable();
					$foreignEntityName = $foreignTable->getClassName();
					$foreignPeerClassName = $foreignTable->getPeerClassName();
					if (method_exists($foreignPeerClassName, 'getAll'))
						$foreignEntities = call_user_func(array($foreignPeerClassName, 'getAll'));
					$pluralizedEntityName = Common::pluralize(Common::strtocamel($foreignEntityName, false));
					$this->smarty->assign($pluralizedEntityName, $foreignEntities);
				}
			}
		}
	}

	/**
	 * Respuesta generica de errores luego de un DoEdit
	 * @param $mapping mapping del phpmvc-config
	 * @param $smarty instancia de smarty
	 * @param $object objeto que se queria guardar
	 * @param $forward forward
	 */
	protected function returnFailure($mapping, $smarty, $object, $forward) {

		$objectName = lcfirst(get_class($object));
		$this->smarty->assign($objectName, $object);

		$id = $object->getId();
		if (empty($id))
			$this->smarty->assign('action', 'create');
		else
			$this->smarty->assign('action', 'edit');

		$this->smarty->assign('message', 'error');
		return $mapping->findForwardConfig($forward);
	}
	
	/**
	 * Makes an ajax request fail
	 */
	function returnAjaxFailure($msg = 'Internal Server Error') {
		header('HTTP/1.1 500 '.$msg);
	}

	/**
	 * Indica si una accion fu invocadamediante ajax
	 * @returns bool si fue invocada via ajax true, si no, false
	 */
	public function isAjax() {
		return (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && ($_SERVER['HTTP_X_REQUESTED_WITH'] == 'XMLHttpRequest'));
	}

	/**
	 * Agrega parametros a un url
	 * @param $url string con url
	 * @param $params array con parametros a agregar a un url
	 * @returns url mas parametros
	 */
	function addParams($url, $params) {
		$urlWithParams = $url;
		foreach ($params as $param => $value) {
			$urlWithParams .= '&'.$param.'='.$value;
		}
		return $urlWithParams;
	}

	/**
	 * Elimina determinados parametros de una url
	 * @param $url string con url
	 * @param $paramsForRemoval array con parametros a remover a un url
	 * @returns url sin parametrosmas parametros
	 */
	function removeParams($url, $paramsForRemoval) {
		$temp = preg_split('/\?/', $url);
		$prefix = $temp[0];
		$oldParams = $temp[1];
		$oldNameValuePairs = preg_split('/&/', $oldParams);
		$newNameValuePairs = array();
		foreach ($oldNameValuePairs as $nameValuePair) {
			$markedForRemoval = false;
			foreach ($paramsForRemoval as $paramForRemoval) {
				$aux = preg_split('/=/', $nameValuePair);
				if ($paramForRemoval == $aux[0])
					$markedForRemoval = true;
			}
			if (!$markedForRemoval)
				array_push($newNameValuePairs, $nameValuePair);
		}
		$newUrl = $prefix;
		$isFirst = true;
		foreach ($newNameValuePairs as $nameValuePair) {
			if ($isFirst) {
				$newUrl .= '?';
				$isFirst = false;
			}
			else
				$newUrl .= '&';

			$newUrl .= $nameValuePair;
		}
		return $newUrl;
	}

	/**
	 * Genera un forward dinamico a partir de un forward existente agregando o sacando parametros a la url
	 * @param $forwardName nombre del forward
	 * @param array $addParams parametros a agregar a la url
	 * @param array $removeParams parametros a remover a la url
	 * @returns ForwardConfig generado
	 */
	function generateDynamicForward($forwardName, $addParams = array(), $removeParams = array()) {
		switch ($forwardName) {
			case 'success':
				//$action = $this->getAction($_SERVER['HTTP_REFERER']);
				$url = $_SERVER['HTTP_REFERER'];
				$url = $this->removeParams($url, $removeParams);
				$url = $this->addParams($url, $addParams);
				/*return new ForwardConfig($this->addParams($addParams, 
					'Main.php?do='.$action), True);*/
				return new ForwardConfig($url, True);
			case 'failure':
				//$action = $this->getAction($_SERVER['HTTP_REFERER']);
				$url = $_SERVER['HTTP_REFERER'];
				$url = $this->removeParams($url, $removeParams);
				$url = $this->addParams($url, $addParams);
				/*return new ForwardConfig($this->addParams($addParams, 
					'Main.php?do='.$action), True);*/
				return new ForwardConfig($url, True);
			default:
				throw new Exception('invalid argument "'.$forwardName.'" for '.$forwardName);
		}
	}	

	function getDependencyId() {
		if (Common::isAffiliatedUser())
			$dependencyId = Common::getAffiliatedId();
		else {
			if (!empty($_GET['dependencyId']))
				$dependencyId = $_GET['dependencyId'];
			else {
				if (!empty($_GET['objectiveId'])) {
					require_once('TableroObjectivePeer.php');
					$objective = TableroObjectivePeer::get($_GET['objectiveId']);
					$dependencyId = $objective->getAffiliateId();
				}
				if (!empty($_GET['projectId'])) {
					require_once('TableroProjectPeer.php');
					require_once('TableroObjectivePeer.php');
					$project = TableroProjectPeer::get($_GET['projectId']);
					$objective = $project->getTableroObjective();
					$dependencyId = $objective->getAffiliateId();
				}
			}
		}
		return $dependencyId;
	}


} // BaseAction
