<?php

namespace Egytca\Propel;

use BasePeer;
use Common;
use Criteria;
use Exception;
use ModelCriteria;

class Query extends ModelCriteria {

	private static $MAGIC_METHODS = array('findBy', 'findOneBy', 'filterBy', 'orderBy', 'groupBy');

	function __call($name, $arguments) {

		if (preg_match("/^with(.+)column/i", $name, $matches)) {
			$columnName = $matches[1];
			$sqlMethod = 'sqlFor'.$columnName.'Column';
			if (method_exists($this, $sqlMethod)) {
				$sql = call_user_func_array([$this, $sqlMethod], $arguments);
				return $this->withColumn($sql, $columnName);
			}
		}

		return parent::__call($name, $arguments);
	}

	/**
	 * Permite agregar un filtro a la Query, este filtro puede ser
	 * o bien un campo del modelo o bien un filtro personalizado.
	 *
	 * @param  string $filterName
	 * @param  mixed $filterValue
	 * @return Query
	 */
	public function addFilter($filterName, $filterValue) {

		// Si el $filterName existe como metodo en el objeto query, lo invoca.
		$result = $this->callIfPossible($filterName, array($filterValue));
		if ($result instanceof ModelCriteria)
			return $this;

		switch ($filterName) {

			case 'searchString':
				$this->filterByName("%$filterValue%", Criteria::LIKE);
				break;

			case 'notInIds':
				$this->filterById($filterValue, Criteria::NOT_IN);
				break;

			case 'entityFilter':
				$this->entityFilter($filterValue);
				break;

			case 'dateRange':
				foreach ($filterValue as $rangeName => $rangeValue) {
					$rangeMinMax = Common::getPeriodArray(
						$rangeValue['min'],
						$rangeValue['max']
					);
					if (!empty($rangeMinMax))
						$this->addFilter($rangeName, $rangeMinMax);
				}
				break;

			default:
				$peer = $this->getModelPeerName();
				$filterName = ucfirst(strtolower($filterName));
				if (in_array($filterName, $peer::getFieldNames(BasePeer::TYPE_PHPNAME))) {
					// filterByXxx acepta arrays. filterBy('Xxx', $v) no
					$filterMethod = 'filterBy'.$filterName;
					$this->$filterMethod($filterValue);
				} elseif (is_array($filterValue))
					$this->addFilters($filterValue);
				else {
					// global $system;
					// if (($system['config']['system']['parameters']['debugMode']['value'] == 'YES')) {
					// 	$this->reportMissingFilter($filterName);
					// }
				}

				break;
		}

		return $this;
	}

	private function reportMissingFilter($filterName) {

		$e = new Exception();
		$trace = $e->getTraceAsString();

		$error = "<p>invalid filter '$filterName'</p>";
		$error .= "<pre>";

		$error .= "stack trace:\n$trace\n\n";

		if (isset($_SERVER['HTTP_REFERER']))
			$error .= "referrer: " . urldecode($_SERVER['HTTP_REFERER']) . "\n\n";

		if (isset($_SERVER['REQUEST_URI']))
			$error .= "request: " . urldecode($_SERVER['REQUEST_URI']) . "\n";

		$error .= "</pre>";

		$user = Common::getLoggedUser();
		if (is_object($user))
			$userInfo = $user->getUsername();
		else
			$userInfo = "Visitor";

		global $system;
		$subject = "SITIO: ".$system['config']['system']['parameters']['siteShortName']." / Error generado por ".$userInfo;
		$email = explode(',', $system['config']['system']['parameters']['debugMail']);
		$mailFrom = $system['config']['system']['parameters']['fromEmail'];

		require_once 'EmailManagement.php';
		$mailer = new \EmailManagement();
		$message = $mailer->createHTMLMessage($subject, $error);
		$result = $mailer->sendMessage($email, $mailFrom, $message);
	}

	/**
	 * Agrega multiples filtros a la Query.
	 *
	 * @see    addFilter
	 * @param  array $filters
	 * @return Query query
	 */
	public function addFilters($filters = array()) {
		foreach ($filters as $name => $value) {
			if ( ($value != null || $value == '0') && $name != "perPage")
				$this->addFilter($name, $value);
		}
		return $this;
	}

	/**
	 * De existir, invoca al $method de la query.
	 *
	 * @param  string $method
	 * @param  mixed $arguments
	 * @return mixed
	 */
	private function callIfPossible($method, $arguments) {

		if (method_exists($this, $method) || ($this->isMagicMethod($method))) {
			return call_user_func_array(array($this, $method), $arguments);
		}
		return false;
	}

	/**
	 * Se fija si $method es un metodo magico.
	 *
	 * @param  string $method
	 * @return boolean
	 */
	private function isMagicMethod($method) {
		foreach (self::$MAGIC_METHODS as $magicMethod) {
			if (preg_match("/^$magicMethod/", $method))
				return true;
		}
		return false;
	}

	/**
	 * Crea un pager.
	 *
	 * @param  array $filters
	 * @param  int $page
	 * @param  int $perPage
	 * @return PropelModelPager
	 */
	public function createPager($filters, $page = 1, $perPage = 10) {
		return $this->addFilters($filters)->paginate($page, $perPage);
	}

	/**
	 * Filtro por entidad asociada
	 *
	 * @param array $filterValue
	 */
	public function entityFilter($filterValue) {

		$entityQueryClass = ucfirst($filterValue['entityType']) . "Query";
		if (!class_exists(ucfirst($filterValue['entityType'])) || !class_exists($entityQueryClass))
			return; // nothing to filter

		$entity = $entityQueryClass::create()->findOneById($filterValue['entityId']);

		if ($entity) {
			$filterByEntity = 'filterBy'.ucfirst($filterValue['entityType']);

			$queryClass = get_class($this);

			if ($filterValue['getCandidates']) {
				$ids = $queryClass::create()->select("Id")->$filterByEntity($entity)->find()->toArray();
				$this->filterById($ids, Criteria::NOT_IN);
			} else {
				$this->$filterByEntity($entity);
			}
		}

	}
}
