<?php

/**
 * BaseListAction
 *
 * Meta clase para obtener una coleccion (listado) de objetos en la base de datos
 *
 * @package phpMVC
 * @author Modulos Empresarios / Egytca
 * @copyright Copyright (c) 2015, Egytca
 */

class BaseListAction extends BaseAction {

	/**
	 * Nombre de la clase del objeto que se va a obtener
	 * @var        string
	 */
	private $entityClassName;

	/**
	 * Objeto Smarty
	 * @var        smarty
	 */
	protected $smarty;

	/**
	 * Nombre del template si es llamado por ajax
	 * @var        string
	 */
	protected $ajaxTemplate;

	/**
	 * Array de filtros pasados por $_GET
	 * @var        array
	 */
	protected $filters;

	/**
	 * Objeto de consulta (query) a la base de datos
	 * @var        query
	 */
	protected $query;

	/**
	 * Objetos obtenidos de ejecutar la de consulta (query) a la base de datos
	 * @var        PropelObjectCollection
	 */
	protected $results;

	/**
	 * Objeto paginador para generar paginador
	 * @var        PropelModelPager
	 */
	protected $pager;

	/**
	 * Numero de pagina que se esta solicitando
	 * @var        integer
	 */
	protected $page;

	/**
	 * Cantidad de objetos por pagina
	 * @var        integer
	 */
	protected $perPage;

	/**
	 * Indica si se usa paginador o no
	 * @var        boolean
	 */
	protected $notPaginated = false;

	/**
	 * Nombre del parametro forward en caso de ejecutarse bien
	 * @var        string
	 */
	protected $forwardName = 'success';

	/**
	 * Nombre del parametro forward en caso de ejecutarse con error
	 * @var        string
	 */
	protected $forwardFailureName = 'failure';

	/**
	 * Indica si usa la salida a Xls
	 * @var        boolean
	 */
	protected $toXls = false;

	/**
	 * Indica si esta permitido asignar $this->toXls desde la request
	 * @var        boolean
	 */
	protected $allowXlsParam = false;

	/**
	 * Constructor
	 *
	 * @param string $entityClassName Nombre de la clase que representa la entidad que se va a listar
	 */
	function __construct($entityClassName) {

		if (empty($entityClassName))
			throw new Exception('$entityClassName must be set');

		if (!class_exists($entityClassName))
			throw new Exception('Class "' . $entityClassName .'" not found or unavailable');

		$this->entityClassName = $entityClassName;

		if (substr(get_class($this), -7, 1) != 'X')
			$this->ajaxTemplate = str_replace('Action', '', get_class($this)).'X.tpl';
		else
			$this->ajaxTemplate = str_replace('Action', '', get_class($this)) . '.tpl';

	}

	/**
	 * Execute
	 * @see BaseAction::execute()
	 */
	function execute($mapping, $form, &$request, &$response) {

		parent::execute($mapping, $form, $request, $response);

		// Verificamos la existencia de la clase de la que se obtendra la coleccion
		if (class_exists($this->entityClassName)) {

			$this->query = $this->createQuery();

			// Acciones a ejecutar antes de obtener la coleccion de objetos
			// Si el preList devuelve false, se retorna $mapping->findForwardConfig($this->forwardFailureName)
			if ($this->preList() === false)
				return $mapping->findForwardConfig($this->forwardFailureName);

			if (!$this->notPaginated) {

				if (!isset($this->perPage) || $this->perPage < 1)
					$this->perPage = Common::getRowsPerPage($this->module);

				$this->page = $request->getParameter('page');
				$this->pager = $this->query->createPager($this->filters, $this->page, $this->perPage);

				$this->results = $this->pager->getResults();
				$this->smarty->assign('pager', $this->pager);
			}
			else
				$this->results = $this->query->setFormatter(ModelCriteria::FORMAT_ON_DEMAND)->addFilters($this->filters)->find();

			// Genera info de debug si esta presente
			if (isset($this->debugData)) {
				$this->debugData->set(get_class($this).' filters', $this->filters);
				$this->debugData->set(get_class($this).' main query ('.get_class($this->query).')', $this->query->toString());
			}

			// Acciones a ejecutar despues de obtener la coleccion de objetos
			$this->postList();

			$this->smarty->assign(lcfirst($this->entityClassName) . 'Coll', $this->results);

			if ($this->toXls) { // Exportar a Xls

				$this->template->template = "TemplateAjax.tpl";

				$xml = $this->smarty->fetch(str_replace('Action', '', get_class($this)) . 'Xml.tpl');

				require_once("ExcelManagement.php");
				$excel = new ExcelManagement();
				$excel->sendXlsFromXml($xml);
				return;

			} else {
				return $mapping->findForwardConfig($this->forwardName);
			}
		}
		else {
			$this->errors->add('Class "'.$this->entityClassName.'" does not exist.');
			$this->smarty->assign('message', 'An error occurred.');
			return $mapping->findForwardConfig($this->forwardFailureName);
		}
	}

	protected function createQuery() {
		return BaseQuery::create($this->entityClassName);
	}


/**
 * preList
 * Acciones a ejecutar antes de obtener la coleccion de objetos
 */
	protected function preList() {

		if ($this->allowXlsParam && $_GET['toXls']) {
			$this->toXls = true;
		}

		if ($this->toXls) {
			$this->notPaginated = true;
		}

		// Procesar los valores del request asociados a filtros
		$this->prepareFilters();

		return true;
	}

/**
 * postList
 * Acciones a ejecutar despues de obtener la coleccion de objetos
 */
	protected function postList() {

		// Informacion para armar los links de paginador
		$this->assignFiltersToSmarty();

		$this->smarty->assign('entityClass', $this->entityClassName);
		$this->smarty->assign('entityColl', $this->results);

		return true;
	}

}
