<?php

/**
 * Pruebas temporales hasta que se puedan usar las pruebas unitarias de Codeception
 * Uso:
 *     <?php
 *     $runTests = require('WEB-INF/classes/includes/Egytca/ParamChecker/Tests.php');
 *     $runTests();
 */

namespace Egytca\ParamChecker;

$_POST = [
	'emptyString' => '',
	'zeroString' => '0',
	'arrayOfEmptyStrings' => ['', '']
];

/*
echo "<p>\$_GET</p>";
echo "<pre>";
var_dump($_GET);
echo "</pre>";

echo "<p>\$_POST</p>";
echo "<pre>";
var_dump($_POST);
echo "</pre>";
*/

$paramChecker = new ParamChecker([
	'method' => 'POST'
]);

$passed = function($result) {
	return count($result->getErrors()) == 0;
};

$notPassed = function($result) {
	return count($result->getErrors()) > 0;
};

$printResult = function($result) {
	echo '<p>errors</p>';
	echo '<pre>';
	print_r($result->getErrors());
	echo '</pre>';
	echo '<p>params</p>';
	echo '<pre>';
	print_r($result->getParams());
	echo '</pre>';
};

$tests = [
	'testNotSetAndRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'notSet',
			'required' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$notPassed($result),
			'a param that is not set should not pass "required"'
		);
		$assert(
			count($result->getParams()) == 0,
			'should not have accepted any parameters'
		);
	},

	'testNotSetNorRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'notSet',
			'required' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'a param that is not set without "required" should pass'
		);
		$assert(
			count($result->getParams()) == 0,
			'should not have accepted any parameters'
		);
	},

	'testEmptyStringRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'emptyString',
			'required' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'empty string should pass "required"'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('emptyString') === null,
			'should have accepted empty string parameter as null'
		);
	},

	'testEmptyStringNotRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'emptyString',
			'required' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'empty string without "required" should pass'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('emptyString') === null,
			'should have accepted empty string parameter as null'
		);
	},

	'testZeroStringAndRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'zeroString',
			'required' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'zero string should pass "required"'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('zeroString') === '0',
			'should have accepted zero string parameter as "0"'
		);
	},

	'testZeroStringNotRequired' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'zeroString',
			'required' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'zero string without "required" should pass'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('zeroString') === '0',
			'should have accepted zero string parameter as "0"'
		);
	},

	'testEmptyStringNotNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'emptyString',
			'nonEmpty' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'empty string without "nonEmpty" should pass'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('emptyString') === null,
			'should have accepted empty string parameter as null'
		);
	},

	'testEmptyStringWithNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'emptyString',
			'nonEmpty' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$notPassed($result),
			'empty string should not pass "nonEmpty"'
		);
		$assert(
			count($result->getParams()) == 0,
			'should not have accepted any parameters'
		);
	},

	'testZeroStringAndNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'zeroString',
			'nonEmpty' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'zero string should pass "nonEmpty"'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('zeroString') === '0',
			'should have accepted zero string parameter as "0"'
		);
	},

	'testZeroStringNotNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'zeroString',
			'nonEmpty' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'zero string without "nonEmpty" should pass'
		);
		$assert(
			count($result->getParams()) == 1 && $result->getParam('zeroString') === '0',
			'should have accepted zero string parameter as "0"'
		);
	},

	'testArrayOfEmptyStringsWithNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'arrayOfEmptyStrings',
			'nonEmpty' => true
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'array of empty strings should pass "nonEmpty"'
		);
		$assert(
			count($result->getParams()) == 1 && is_array($result->getParam('arrayOfEmptyStrings')),
			'should have accepted array of empty strings as array'
		);
	},

	'testArrayOfEmptyStringsWithoutNonEmpty' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$desc = [[
			'name' => 'arrayOfEmptyStrings',
			'nonEmpty' => false
		]];
		$result = $paramChecker->check($desc);
		$assert(
			$passed($result),
			'array of empty strings without "nonEmpty" should pass'
		);
		$assert(
			count($result->getParams()) == 1 && is_array($result->getParam('arrayOfEmptyStrings')),
			'should have accepted array of empty strings as array'
		);
	},

	'testFailedParamDoesNotBlockOthers' => function($assert) use ($paramChecker, $passed, $notPassed, $printResult) {
		$_POST = ['falseNumber' => 'house123', 'anInteger' => '123'];
		$desc = [
			[ 'name' => 'falseNumber', 'conditions' => ['numeric'] ],
			[ 'name' => 'anInteger', 'conditions' => ['integer'] ]
		];
		$result = $paramChecker->check($desc);
		$assert(
			count($result->getErrors()) == 1,
			'falseNumber shoud not pass'
		);
		$assert(
			( count($result->getParams()) == 1 ) && ( $result->getParam('anInteger') == 123 ),
			'param "anInteger" should have been accepted'
		);
		//$printResult($result);
	}

];

$run = function() use ($tests) {
	echo "<ul>";
	$count = 1;
	foreach ($tests as $name => $test) {
		$errorsFound = false;
		$assert = function($cond, $errorMsg) use ($count, $name, &$errorsFound) {
			if (!$cond) {
				echo "<li>$count - $name: <span style=\"background: pink;\">ERROR</span> $errorMsg</li>";
				$errorsFound = true;
			}
		};
		$test($assert);
		if (!$errorsFound)
			echo "<li>$count - $name: <span style=\"background: lightgreen;\">OK</span></li>";

		$count++;
	}
	echo "</ul>";
};

return $run;